<?php
// Enable full error reporting for debugging (remove in production)
error_reporting(E_ALL);
ini_set('display_errors', '1');
ini_set('display_startup_errors', '1');
ini_set('log_errors', '1');
ini_set('error_log', __DIR__ . '/../var/log/php_errors.log');

require_once __DIR__ . '/../vendor/autoload.php';
// load procedural DB helper (contains App\get_db())
require_once __DIR__ . '/../src/db.php';

use App\Product;
use App\User;
use App\ClientPanel;
use function App\get_db;

// Session cookie without persistent lifetime (browser session only).
session_set_cookie_params([
    'lifetime' => 0,
    'path' => '/',
    'secure' => (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off'),
    'httponly' => true,
    'samesite' => 'Lax',
]);
session_start();

// Auto-close authenticated session after inactivity.
$sessionTimeout = 1800; // 30 minutes
if (isset($_SESSION['last_activity']) && (time() - (int)$_SESSION['last_activity']) > $sessionTimeout) {
    unset($_SESSION['user_id']);
}
$_SESSION['last_activity'] = time();

// Load .env if present (used later too)
$env = @parse_ini_file(__DIR__ . '/../.env', INI_SCANNER_RAW) ?: [];
$configuredBasePath = isset($env['BASE_PATH']) ? $env['BASE_PATH'] : (getenv('BASE_PATH') ?: null);
$prettyRoutesFlag = isset($env['PRETTY_URLS']) ? $env['PRETTY_URLS'] : (getenv('PRETTY_URLS') ?: '0');
$usePrettyRoutes = in_array(strtolower((string)$prettyRoutesFlag), ['1', 'true', 'yes', 'on'], true);

// Normalize request path and strip any base directory (useful when not at webroot)
$path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
if ($configuredBasePath) {
    $basePath = $configuredBasePath;
} else {
    $basePath = str_replace('\\', '/', dirname($_SERVER['SCRIPT_NAME']));
}

// Normalize basePath to a canonical form (leading slash, no trailing slash unless root)
$basePath = '/' . trim($basePath, '/');
if ($basePath === '/.') { $basePath = '/'; }

// Remove basePath from the beginning of $path if present
if ($basePath !== '/' && strpos($path, $basePath) === 0) {
    $path = substr($path, strlen($basePath));
}

// Ensure path starts with /
if (strpos($path, '/') !== 0) {
    $path = '/' . $path;
}

// Remove trailing slash (except for root)
if ($path !== '/' && substr($path, -1) === '/') {
    $path = rtrim($path, '/');
}

// When accessing /index.php directly without route param, treat it as home.
if ($path === '/index.php') {
    $path = '/';
}

// Base path used by templates and redirects (empty when app runs at domain root)
$appBasePath = $basePath === '/' ? '' : $basePath;
$url = static function ($route = '', array $query = []) use ($appBasePath, $usePrettyRoutes) {
    $route = '/' . ltrim((string)$route, '/');
    $isStaticAsset = strpos($route, '/assets/') === 0;
    if ($usePrettyRoutes) {
        $target = $route === '/' ? ($appBasePath ?: '/') : $appBasePath . $route;
        if (!empty($query)) {
            $target .= '?' . http_build_query($query);
        }
        return $target;
    }

    if ($isStaticAsset) {
        $target = $appBasePath . $route;
        if (!empty($query)) {
            $target .= '?' . http_build_query($query);
        }
        return $target;
    }

    $target = ($appBasePath ?: '') . '/index.php';
    if ($route !== '/') {
        $query['route'] = ltrim($route, '/');
    }
    if (!empty($query)) {
        $target .= '?' . http_build_query($query);
    }
    return $target;
};

// Query-string route fallback for servers without mod_rewrite enabled
$routeFromQuery = trim((string)($_GET['route'] ?? ''), '/');
if ($routeFromQuery !== '') {
    $path = '/' . $routeFromQuery;
}

$currentUser = null;
$accountAccess = null;
if (isset($_SESSION['user_id'])) {
    $currentUser = User::find((int)$_SESSION['user_id']);
    if (!$currentUser) {
        unset($_SESSION['user_id']);
    } else {
        $accountAccess = User::accountAccessStatus($currentUser);
    }
}

$getCartKey = static function ($user) {
    if (is_array($user) && isset($user['id'])) {
        return 'cart_user_' . (int)$user['id'];
    }
    return 'cart_guest';
};

$mergeGuestCartIntoUser = static function (int $userId) {
    $guestKey = 'cart_guest';
    $userKey = 'cart_user_' . $userId;
    $guestCart = $_SESSION[$guestKey] ?? [];
    if (!is_array($guestCart) || empty($guestCart)) {
        return;
    }
    $userCart = $_SESSION[$userKey] ?? [];
    if (!is_array($userCart)) {
        $userCart = [];
    }
    foreach ($guestCart as $pid => $qty) {
        $pid = (int)$pid;
        $qty = max(0, (int)$qty);
        if ($pid <= 0 || $qty <= 0) {
            continue;
        }
        $userCart[$pid] = ($userCart[$pid] ?? 0) + $qty;
    }
    $_SESSION[$userKey] = $userCart;
    unset($_SESSION[$guestKey]);
};

$activeCartKey = $getCartKey($currentUser);
$selectedProductSessionKey = 'last_selected_product_id';
$billingNotice = (string)($_SESSION['billing_notice'] ?? '');
unset($_SESSION['billing_notice']);
$selectionNotice = (string)($_SESSION['selection_notice'] ?? '');
unset($_SESSION['selection_notice']);
$catalogNotice = (string)($_SESSION['catalog_notice'] ?? '');
unset($_SESSION['catalog_notice']);

$requireTrialAccess = static function () use ($accountAccess, $url) {
    if (!empty($accountAccess['is_locked'])) {
        $_SESSION['billing_notice'] = 'Tu prueba de 4 días terminó. Para seguir gestionando evento, invitados y envíos, realiza tu pago.';
        header('Location: ' . $url('/dashboard'));
        exit;
    }
};

$requireSelectedInvitation = static function () use ($currentUser, $selectedProductSessionKey, $url) {
    if (!$currentUser) {
        return;
    }

    $selectedProductId = (int)($_SESSION[$selectedProductSessionKey] ?? 0);
    if ($selectedProductId > 0) {
        return;
    }

    $eventData = ClientPanel::getEventByUser((int)$currentUser['id']);
    if (!empty($eventData['source_product_id'])) {
        $_SESSION[$selectedProductSessionKey] = (int)$eventData['source_product_id'];
        return;
    }

    $_SESSION['catalog_notice'] = 'Primero selecciona una invitación del catálogo para habilitar evento e invitados.';
    header('Location: ' . $url('/'));
    exit;
};

if ($path === '/register') {
    $authError = null;
    $next = (string)($_POST['next'] ?? $_GET['next'] ?? '');
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $name = trim((string)($_POST['name'] ?? ''));
        $email = trim((string)($_POST['email'] ?? ''));
        $password = (string)($_POST['password'] ?? '');
        if ($name === '' || $email === '' || strlen($password) < 6) {
            $authError = 'Completa nombre, email y una contraseña de al menos 6 caracteres.';
        } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $authError = 'Email inválido.';
        } elseif (User::findByEmail($email)) {
            $authError = 'Ese email ya está registrado.';
        } else {
            $newId = User::create($name, $email, $password);
            session_regenerate_id(true);
            $_SESSION['user_id'] = $newId;
            $mergeGuestCartIntoUser($newId);
            $redirectRoute = ($next === 'checkout') ? '/cart' : '/dashboard';
            header('Location: ' . $url($redirectRoute)); exit;
        }
    }
    include __DIR__ . '/../templates/register.php';
    exit;
}

if ($path === '/login') {
    $authError = null;
    $next = (string)($_POST['next'] ?? $_GET['next'] ?? '');
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $email = trim((string)($_POST['email'] ?? ''));
        $password = (string)($_POST['password'] ?? '');
        $user = User::verifyCredentials($email, $password);
        if (!$user) {
            $authError = 'Credenciales inválidas.';
        } else {
            session_regenerate_id(true);
            $_SESSION['user_id'] = (int)$user['id'];
            $mergeGuestCartIntoUser((int)$user['id']);
            $redirectRoute = ($next === 'checkout') ? '/cart' : '/dashboard';
            header('Location: ' . $url($redirectRoute)); exit;
        }
    }
    include __DIR__ . '/../templates/login.php';
    exit;
}

if ($path === '/logout') {
    unset($_SESSION['user_id']);
    session_regenerate_id(true);
    header('Location: ' . $url('/')); exit;
}

if ($path === '/switch-user') {
    unset($_SESSION['user_id']);
    header('Location: ' . $url('/login')); exit;
}

if ($path === '/dashboard') {
    if (!$currentUser) {
        header('Location: ' . $url('/login')); exit;
    }
    $orders = \App\Order::allByUser((int)$currentUser['id']);
    $eventData = ClientPanel::getEventByUser((int)$currentUser['id']);
    $eventDetails = (!empty($eventData['id'])) ? ClientPanel::getEventDetailsMap((int)$eventData['id']) : [];
    $selectedInvitation = null;
    $invitePreviewUrl = null;
    if (!empty($eventData['source_product_id'])) {
        $selectedInvitation = Product::find((int)$eventData['source_product_id']);
    }
    if ($selectedInvitation) {
        $pickValue = static function (...$values) {
            foreach ($values as $v) {
                if ($v !== null && trim((string)$v) !== '') {
                    return trim((string)$v);
                }
            }
            return '';
        };
        $resolveImage = static function (?string $rawImage) use ($appBasePath) {
            $rawImage = trim((string)$rawImage);
            if ($rawImage === '') {
                return '';
            }
            if (preg_match('#^(https?:)?//#i', $rawImage) || strpos($rawImage, 'data:') === 0) {
                return $rawImage;
            }
            if (strpos($rawImage, '/') === 0) {
                return $appBasePath . $rawImage;
            }
            return $appBasePath . '/' . ltrim($rawImage, '/');
        };

        $previewQuery = [
            'event_title' => $pickValue($eventData['title'] ?? null, $selectedInvitation['title'] ?? null),
            'main_message' => $pickValue($eventData['main_message'] ?? null, $eventDetails['main_message'] ?? null),
            'main_person_name' => $pickValue($eventData['main_person_name'] ?? null, $eventDetails['main_person_name'] ?? null),
            'secondary_person_name' => $pickValue($eventData['secondary_person_name'] ?? null, $eventDetails['secondary_person_name'] ?? null),
            'event_date' => $pickValue($eventData['event_date'] ?? null, $eventDetails['event_date'] ?? null),
            'event_time' => $pickValue($eventData['event_time'] ?? null, $eventDetails['event_time'] ?? null),
            'salon_name' => $pickValue($eventData['salon_name'] ?? null, $eventData['venue'] ?? null, $eventDetails['salon_name'] ?? null),
            'event_address' => $pickValue($eventData['event_address'] ?? null, $eventDetails['event_address'] ?? null),
            'ceremony_place' => $pickValue($eventData['ceremony_place'] ?? null, $eventDetails['ceremony_place'] ?? null),
            'ceremony_time' => $pickValue($eventData['ceremony_time'] ?? null, $eventDetails['ceremony_time'] ?? null),
            'dress_code' => $pickValue($eventData['dress_code'] ?? null, $eventDetails['dress_code'] ?? null),
            'rsvp_deadline' => $pickValue($eventData['rsvp_deadline'] ?? null, $eventDetails['rsvp_deadline'] ?? null),
            'contact_whatsapp' => $pickValue($eventData['contact_whatsapp'] ?? null, $eventDetails['contact_whatsapp'] ?? null),
            'maps_url' => $pickValue($eventData['maps_url'] ?? null, $eventDetails['maps_url'] ?? null),
            'gift_table' => $pickValue($eventDetails['gift_table'] ?? null),
            'padrinos_names' => $pickValue($eventData['padrinos_names'] ?? null, $eventDetails['padrinos_names'] ?? null),
            'parents_names' => $pickValue($eventDetails['parents_names'] ?? null),
            'rsvp_url' => $url('/enviar-invitaciones'),
        ];

        $productImage = $resolveImage((string)($selectedInvitation['image_url'] ?? ''));
        if ($productImage !== '') {
            $previewQuery['photo_main'] = $productImage;
            $previewQuery['photo_ceremony'] = $productImage;
            $previewQuery['photo_gallery1'] = $productImage;
        }

        $previewQuery = array_filter($previewQuery, static function ($v) {
            return trim((string)$v) !== '';
        });

        $invitePreviewUrl = $appBasePath . '/invitaciones/floral-sobre-tech/index.html';
        if (!empty($previewQuery)) {
            $invitePreviewUrl .= '?' . http_build_query($previewQuery);
        }
    }
    include __DIR__ . '/../templates/dashboard.php';
    exit;
}

if ($path === '/organizar-evento') {
    if (!$currentUser) {
        header('Location: ' . $url('/login')); exit;
    }
    $requireSelectedInvitation();
    $requireTrialAccess();
    $plannerMsg = null;
    $eventTypes = ClientPanel::listEventTypes();
    $eventTypesMap = [];
    foreach ($eventTypes as $t) {
        $eventTypesMap[(string)$t['code']] = (string)$t['label'];
    }
    $eventData = ClientPanel::getEventByUser((int)$currentUser['id']);
    $selectedProductId = (int)($_SESSION[$selectedProductSessionKey] ?? 0);
    if ($selectedProductId <= 0 && !empty($eventData['source_product_id'])) {
        $selectedProductId = (int)$eventData['source_product_id'];
    }
    $selectedProduct = $selectedProductId > 0 ? Product::find($selectedProductId) : null;
    if (!$selectedProduct) {
        $cartSnapshot = $_SESSION[$activeCartKey] ?? [];
        if (is_array($cartSnapshot) && !empty($cartSnapshot)) {
            $fallbackProductId = (int)array_key_first($cartSnapshot);
            $selectedProduct = $fallbackProductId > 0 ? Product::find($fallbackProductId) : null;
        }
    }
    $inferredEventTypeCode = $selectedProduct ? Product::inferEventTypeCode($selectedProduct) : null;
    $resolvedEventTypeCode = $inferredEventTypeCode ?: (string)($eventData['event_type_code'] ?? '');
    $resolvedEventTypeLabel = $resolvedEventTypeCode !== '' ? ($eventTypesMap[$resolvedEventTypeCode] ?? strtoupper($resolvedEventTypeCode)) : '';
    $dynamicFields = $resolvedEventTypeCode !== '' ? ClientPanel::getEventFieldTemplatesByTypeCode($resolvedEventTypeCode) : [];
    $eventDetails = !empty($eventData['id']) ? ClientPanel::getEventDetailsMap((int)$eventData['id']) : [];

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $eventTypeCode = $resolvedEventTypeCode !== '' ? $resolvedEventTypeCode : trim((string)($_POST['event_type_code'] ?? ''));
        $title = trim((string)($_POST['title'] ?? ''));
        $eventDate = trim((string)($_POST['event_date'] ?? ''));
        $venue = trim((string)($_POST['venue'] ?? ''));
        $notes = trim((string)($_POST['notes'] ?? ''));
        $postedDetails = (array)($_POST['details'] ?? []);
        if ($eventTypeCode === '' || $title === '') {
            $plannerMsg = ['type' => 'error', 'text' => 'Selecciona el tipo de invitación y captura el título del evento.'];
        } else {
            ClientPanel::saveEvent(
                (int)$currentUser['id'],
                $eventTypeCode,
                $title,
                $eventDate !== '' ? $eventDate : null,
                $venue !== '' ? $venue : null,
                $notes !== '' ? $notes : null,
                $selectedProduct ? (int)$selectedProduct['id'] : null
            );
            $eventData = ClientPanel::getEventByUser((int)$currentUser['id']);
            $resolvedEventTypeCode = (string)($eventData['event_type_code'] ?? $eventTypeCode);
            $dynamicFields = $resolvedEventTypeCode !== '' ? ClientPanel::getEventFieldTemplatesByTypeCode($resolvedEventTypeCode) : [];

            $requiredErrors = [];
            $normalizedDetails = [];
            foreach ($dynamicFields as $field) {
                $fieldKey = (string)($field['field_key'] ?? '');
                if ($fieldKey === '') {
                    continue;
                }
                $value = trim((string)($postedDetails[$fieldKey] ?? ''));
                $normalizedDetails[$fieldKey] = $value;
                if ((int)($field['is_required'] ?? 0) === 1 && $value === '') {
                    $requiredErrors[] = (string)($field['field_label'] ?? $fieldKey);
                }
            }

            if (!empty($requiredErrors)) {
                $plannerMsg = ['type' => 'error', 'text' => 'Faltan campos obligatorios: ' . implode(', ', $requiredErrors)];
                $eventDetails = $normalizedDetails;
            } else {
                if (!empty($eventData['id']) && !empty($normalizedDetails)) {
                    ClientPanel::saveEventDetails((int)$eventData['id'], $normalizedDetails);
                }
                $plannerMsg = ['type' => 'ok', 'text' => 'Evento guardado correctamente.'];
            }
        }
        $eventData = ClientPanel::getEventByUser((int)$currentUser['id']);
        $resolvedEventTypeCode = (string)($eventData['event_type_code'] ?? $eventTypeCode);
        $resolvedEventTypeLabel = $resolvedEventTypeCode !== '' ? ($eventTypesMap[$resolvedEventTypeCode] ?? strtoupper($resolvedEventTypeCode)) : '';
        if (empty($eventDetails) && !empty($eventData['id'])) {
            $eventDetails = ClientPanel::getEventDetailsMap((int)$eventData['id']);
        }
    } elseif (!empty($eventData['id'])) {
        $eventDetails = ClientPanel::getEventDetailsMap((int)$eventData['id']);
    }
    include __DIR__ . '/../templates/event_planner.php';
    exit;
}

if ($path === '/invitados') {
    if (!$currentUser) {
        header('Location: ' . $url('/login')); exit;
    }
    $requireSelectedInvitation();
    $requireTrialAccess();
    $guestMsg = null;
    $eventData = ClientPanel::getEventByUser((int)$currentUser['id']);
    $eventId = $eventData ? (int)$eventData['id'] : null;

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        if (!$eventId) {
            $guestMsg = ['type' => 'error', 'text' => 'Primero selecciona invitación y configura tu evento en Organizar Evento.'];
        }
        $mode = (string)($_POST['mode'] ?? 'manual');
        if ($eventId && $mode === 'manual') {
            $name = trim((string)($_POST['name'] ?? ''));
            $phone = trim((string)($_POST['phone'] ?? ''));
            $email = trim((string)($_POST['email'] ?? ''));
            $adults = (int)($_POST['adults'] ?? 1);
            $kids = (int)($_POST['kids'] ?? 0);
            if ($name === '') {
                $guestMsg = ['type' => 'error', 'text' => 'El nombre del invitado es obligatorio.'];
            } else {
                ClientPanel::addGuest((int)$currentUser['id'], $eventId, $name, $phone !== '' ? $phone : null, $email !== '' ? $email : null, $adults, $kids);
                $guestMsg = ['type' => 'ok', 'text' => 'Invitado agregado correctamente.'];
            }
        } elseif ($eventId && $mode === 'csv' && isset($_FILES['csv']) && is_uploaded_file($_FILES['csv']['tmp_name'])) {
            $result = ClientPanel::importGuestsFromCsvFile((int)$currentUser['id'], $eventId, $_FILES['csv']['tmp_name']);
            if (!empty($result['errors'])) {
                $guestMsg = ['type' => 'error', 'text' => 'Importados: ' . (int)$result['imported'] . '. Errores: ' . implode(' | ', $result['errors'])];
            } else {
                $guestMsg = ['type' => 'ok', 'text' => 'Invitados importados: ' . (int)$result['imported']];
            }
        }
    }

    $guests = ClientPanel::listGuestsByUser((int)$currentUser['id']);
    include __DIR__ . '/../templates/guests.php';
    exit;
}

if ($path === '/enviar-invitaciones') {
    if (!$currentUser) {
        header('Location: ' . $url('/login')); exit;
    }
    $requireSelectedInvitation();
    $requireTrialAccess();
    $sendMsg = null;
    $eventData = ClientPanel::getEventByUser((int)$currentUser['id']);
    $eventId = $eventData ? (int)$eventData['id'] : null;

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        if (!$eventId) {
            $sendMsg = ['type' => 'error', 'text' => 'Primero configura tu evento.'];
        } else {
            $updated = ClientPanel::markGuestsAsInvited((int)$currentUser['id'], $eventId);
            $sendMsg = ['type' => 'ok', 'text' => 'Invitaciones marcadas como enviadas: ' . $updated];
        }
    }

    $guests = ClientPanel::listGuestsByUser((int)$currentUser['id']);
    include __DIR__ . '/../templates/send_invites.php';
    exit;
}

if ($path === '/' || $path === '') {
    $products = Product::all();
    include __DIR__ . '/../templates/list.php';
    exit;
}

if ($path === '/product') {
    $id = (int)($_GET['id'] ?? 0);
    $product = Product::find($id);
    if (!$product) { http_response_code(404); echo 'Producto no encontrado'; exit; }
    include __DIR__ . '/../templates/product.php';
    exit;
}

if ($path === '/cart') {
    $cart = $_SESSION[$activeCartKey] ?? [];
    $cartProducts = Product::findMany(array_keys((array)$cart));
    $checkoutRequiresAuth = empty($currentUser);
    include __DIR__ . '/../templates/cart.php';
    exit;
}

if ($path === '/seleccionar-invitacion' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!$currentUser) {
        header('Location: ' . $url('/login')); exit;
    }
    $requireTrialAccess();
    $id = (int)($_POST['id'] ?? 0);
    $selectedProduct = $id > 0 ? Product::find($id) : null;
    if (!$selectedProduct) {
        $_SESSION['catalog_notice'] = 'No se encontró la invitación seleccionada.';
        header('Location: ' . $url('/'));
        exit;
    }

    $_SESSION[$selectedProductSessionKey] = $id;
    ClientPanel::selectInvitationForUser((int)$currentUser['id'], $selectedProduct);
    $cart = $_SESSION[$activeCartKey] ?? [];
    if (!is_array($cart)) {
        $cart = [];
    }
    $cart[$id] = max(1, (int)($cart[$id] ?? 1));
    $_SESSION[$activeCartKey] = $cart;
    $_SESSION['selection_notice'] = 'Invitación actualizada. Se conservaron los datos de evento e invitados.';
    header('Location: ' . $url('/dashboard'));
    exit;
}

if ($path === '/orders') {
    if (!$currentUser) {
        header('Location: ' . $url('/login')); exit;
    }
    $orders = \App\Order::allByUser((int)$currentUser['id']);
    include __DIR__ . '/../templates/orders.php';
    exit;
}

if ($path === '/add-to-cart' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $id = (int)($_POST['id'] ?? 0);
    $qty = max(1, (int)($_POST['qty'] ?? 1));
    $cart = $_SESSION[$activeCartKey] ?? [];
    if (!is_array($cart)) {
        $cart = [];
    }
    $cart[$id] = ($cart[$id] ?? 0) + $qty;
    $_SESSION[$activeCartKey] = $cart;
    if ($id > 0) {
        $_SESSION[$selectedProductSessionKey] = $id;
        if ($currentUser) {
            $selectedProduct = Product::find($id);
            if ($selectedProduct) {
                ClientPanel::selectInvitationForUser((int)$currentUser['id'], $selectedProduct);
                $_SESSION['selection_notice'] = 'Invitación seleccionada y vinculada a tu evento.';
            }
        }
    }
    if ($currentUser) {
        header('Location: ' . $url('/dashboard')); exit;
    }
    header('Location: ' . $url('/cart')); exit;
}

if ($path === '/remove-from-cart' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $id = (int)($_POST['id'] ?? 0);
    $qtyToRemove = max(1, (int)($_POST['qty'] ?? 1));
    $cart = $_SESSION[$activeCartKey] ?? [];
    if (!is_array($cart)) {
        $cart = [];
    }
    $removedCompletely = false;
    if ($id > 0 && isset($cart[$id])) {
        $currentQty = max(0, (int)$cart[$id]);
        $newQty = $currentQty - $qtyToRemove;
        if ($newQty > 0) {
            $cart[$id] = $newQty;
        } else {
            unset($cart[$id]);
            $removedCompletely = true;
        }
    }
    $_SESSION[$activeCartKey] = $cart;

    if ($currentUser && $removedCompletely && $id > 0 && (int)($_SESSION[$selectedProductSessionKey] ?? 0) === $id) {
        $fallbackProductId = 0;
        foreach (array_keys($cart) as $pid) {
            $pid = (int)$pid;
            if ($pid > 0) {
                $fallbackProductId = $pid;
                break;
            }
        }

        if ($fallbackProductId > 0) {
            $fallbackProduct = Product::find($fallbackProductId);
            if ($fallbackProduct) {
                $_SESSION[$selectedProductSessionKey] = $fallbackProductId;
                ClientPanel::selectInvitationForUser((int)$currentUser['id'], $fallbackProduct);
                $_SESSION['selection_notice'] = 'Se eliminó la invitación y se cambió automáticamente a otra del carrito.';
            }
        } else {
            unset($_SESSION[$selectedProductSessionKey]);
            ClientPanel::clearSelectedInvitationForUser((int)$currentUser['id']);
            $_SESSION['catalog_notice'] = 'El carrito quedó vacío. Selecciona una invitación para habilitar evento e invitados.';
        }
    }

    header('Location: ' . $url('/cart')); exit;
}

if ($path === '/checkout' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!$currentUser) {
        header('Location: ' . $url('/register', ['next' => 'checkout']));
        exit;
    }
    $secret = $env['STRIPE_SECRET'] ?? getenv('STRIPE_SECRET');
    if (!$secret) { echo 'Stripe secret key not configured. Configura STRIPE_SECRET en .env'; exit; }

    \Stripe\Stripe::setApiKey($secret);

    $cart = $_SESSION[$activeCartKey] ?? [];
    $cartProducts = Product::findMany(array_keys((array)$cart));
    $line_items = [];
    foreach ($cart as $pid => $qty) {
        $p = $cartProducts[(int)$pid] ?? null;
        if (!$p) continue;
        $line_items[] = [
            'price_data' => [
                'currency' => 'usd',
                'product_data' => ['name' => $p['title']],
                'unit_amount' => (int)$p['price_cents']
            ],
            'quantity' => $qty
        ];
    }
    if (empty($line_items)) { header('Location: ' . $url('/cart')); exit; }

    $baseUrl = getenv('BASE_URL') ?: ($env['BASE_URL'] ?? null);
    if (!$baseUrl) {
        $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        $baseUrl = $scheme . '://' . $host . $appBasePath;
    }

    $checkout = \Stripe\Checkout\Session::create([
        'payment_method_types' => ['card'],
        'mode' => 'payment',
        'line_items' => $line_items,
        'success_url' => $baseUrl . '/success',
        'cancel_url' => $baseUrl . '/cart',
    ]);

    // compute total amount in cents (fallback)
    $total_cents = 0;
    foreach ($line_items as $li) {
        $unit = $li['price_data']['unit_amount'] ?? 0;
        $qty = $li['quantity'] ?? 1;
        $total_cents += $unit * $qty;
    }

    // record order in DB (pending)
    try {
        $orderId = \App\Order::create($checkout->id, $total_cents, 'pending', $currentUser ? (int)$currentUser['id'] : null);
    } catch (Exception $e) {
        // Log or ignore in this minimal example
    }

    header('Location: ' . $checkout->url);
    exit;
}

if ($path === '/success') {
    // clear only cart and keep authenticated user session
    unset($_SESSION[$activeCartKey]);
    include __DIR__ . '/../templates/success.php';
    exit;
}

if ($path === '/admin') {
    include __DIR__ . '/../admin.php';
    exit;
}

// fallback 404
http_response_code(404);
echo 'Ruta no encontrada';
